﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GAMS;
using System.Reflection;
using System.Data.OleDb;

namespace CutStockGUI
{
    public partial class Form1 : Form
    {
        private List<string> cuts = new List<string>() { "i1", "i2", "i3", "i4" };
        private Dictionary<string, int> d = new Dictionary<string, int>() { { "i1", 97 }, { "i2", 610 }, { "i3", 395 }, { "i4", 211 } };
        private Dictionary<string, int> w = new Dictionary<string, int>() { { "i1", 47 }, { "i2", 36 }, { "i3", 31 }, { "i4", 14 } };
        private int r = 100; // raw width
        private int maxpattern = 35;
        private int nrCuts = 0;
        private List<Color> colors = new List<Color>() { Color.FromArgb(255, 192, 255, 255), Color.FromArgb(255, 255, 255, 192),
                                                         Color.FromArgb(255, 192, 255, 192), Color.FromArgb(255, 255, 224, 192),
                                                         Color.FromArgb(255, 255, 192, 255), Color.FromArgb(255, 129, 255, 192),
                                                         Color.FromArgb(255, 192, 192, 255), Color.FromArgb(255, 255, 192, 192),
                                                         Color.FromArgb(255, 100, 192, 220), Color.FromArgb(255, 129, 100, 192),
                                                         Color.FromArgb(255, 192, 192, 100), Color.FromArgb(255, 255, 192, 100) };

        public Form1()
        {
            InitializeComponent();
            nudWidth.Value = r;
            nudMax.Value = maxpattern;

            // add cut widths
            foreach (string i in cuts)
            {
                flowLayoutPanel1.Controls.Add(new CutWidthsControl(i, w[i], (int) nudWidth.Value, d[i], colors[nrCuts%colors.Count], nudWidth));
                nrCuts++;
            }
        }

        //add new cut width
        private void button3_Click(object sender, EventArgs e)
        {
            flowLayoutPanel1.Controls.Add(new CutWidthsControl("i" + (nrCuts + 1), (int)nudWidth.Value, (int)nudWidth.Value, 0, colors[nrCuts % colors.Count], nudWidth));
            flowLayoutPanel1.VerticalScroll.Value = flowLayoutPanel1.VerticalScroll.Maximum;
            flowLayoutPanel1.Update();
            flowLayoutPanel1.VerticalScroll.Value = flowLayoutPanel1.VerticalScroll.Maximum;
            nrCuts++;
        }

        //delete last cut width
        private void button2_Click(object sender, EventArgs e)
        {
            if (nrCuts <= 1)
                MessageBox.Show("Can't delete last cut width because there has to be at least one.");
            else
            {
                flowLayoutPanel1.Controls.RemoveAt(nrCuts-1);
                flowLayoutPanel1.Update();
                nrCuts--;
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            richTextBox1.Clear();
            tabControl1.Visible = false;
            tabControl1.TabPages.Clear();
            try
            {
                GAMSWorkspace ws = new GAMSWorkspace();
                //GAMSWorkspace ws = new GAMSWorkspace(systemDirectory:System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location));

                // instantiate GAMSOptions and define parameters
                GAMSOptions opt = ws.AddOptions();
                GAMSDatabase cutstockData = ws.AddDatabase("csdata");
                opt.AllModelTypes = "Gurobi";
                opt.OptCR = 0.0; // Solve to optimality
                opt.Defines.Add("pmax", nudMax.Value.ToString());
                opt.Defines.Add("solveMasterAs", "RMIP");

                GAMSSet widths = cutstockData.AddSet("i", 1, "widths");
                GAMSParameter rawWidth = cutstockData.AddParameter("r", 0, "raw width");
                GAMSParameter demand = cutstockData.AddParameter("d", 1, "demand");
                GAMSParameter width = cutstockData.AddParameter("w", 1, "width");

                rawWidth.AddRecord().Value = (int)nudWidth.Value;

                foreach(CutWidthsControl cutWidth in flowLayoutPanel1.Controls)
                {
                    widths.AddRecord(cutWidth.Name);
                    demand.AddRecord(cutWidth.Name).Value = cutWidth.DemandValue;
                    width.AddRecord(cutWidth.Name).Value = cutWidth.WidthValue;
                }

                // create initial checkpoint
                GAMSCheckpoint masterCP = ws.AddCheckpoint();
                GAMSJob masterInitJob = ws.AddJobFromString(GetMasterModel());
                masterInitJob.Run(opt, masterCP, cutstockData);

                GAMSJob masterJob = ws.AddJobFromString("execute_load 'csdata', aip, pp; solve master min z using %solveMasterAs%;", masterCP);

                GAMSSet pattern = cutstockData.AddSet("pp", 1, "pattern index");
                GAMSParameter patternData = cutstockData.AddParameter("aip", 2, "pattern data");

                // Initial pattern: pattern i hold width i
                int patternCount = 0;

                foreach (GAMSParameterRecord rec in width)
                {
                    patternData.AddRecord(rec.Keys[0], pattern.AddRecord((++patternCount).ToString()).Keys[0]).Value = (int)((int)nudWidth.Value / rec.Value);
                }

                // create model instance for sub job
                GAMSCheckpoint subCP = ws.AddCheckpoint();
                ws.AddJobFromString(GetSubModel()).Run(opt, subCP, cutstockData);
                GAMSModelInstance subMI = subCP.AddModelInstance();

                // define modifier demdual
                GAMSParameter demandDual = subMI.SyncDB.AddParameter("demdual", 1, "dual of demand from master");
                subMI.Instantiate("pricing min z using mip", opt, new GAMSModifier(demandDual));

                // find new pattern
                bool patternAdded = true;
                do
                {
                    masterJob.Run(opt, masterCP, cutstockData);
                    // Copy duals into gmssubMI.SyncDB DB
                    demandDual.Clear();
                    foreach (GAMSEquationRecord dem in masterJob.OutDB.GetEquation("demand"))
                        demandDual.AddRecord(dem.Keys[0]).Value = dem.Marginal;

                    subMI.Solve();
                    if (subMI.SyncDB.GetVariable("z").FindRecord().Level < -0.00001)
                    {
                        if (patternCount == maxpattern)
                        {
                            richTextBox1.AppendText("Out of pattern. Increase maxpattern (currently " + nudMax.Value + ")." + Environment.NewLine);
                            patternAdded = false;
                        }
                        else
                        {
                            richTextBox1.AppendText("New pattern! Value: " + subMI.SyncDB.GetVariable("z").FindRecord().Level + Environment.NewLine);
                            richTextBox1.ScrollToCaret();
                            GAMSSetRecord s = pattern.AddRecord((++patternCount).ToString());
                            foreach (GAMSVariableRecord y in subMI.SyncDB.GetVariable("y"))
                            {
                                if (y.Level > 0.5)
                                {
                                    patternData.AddRecord(y.Keys[0], s.Keys[0]).Value = Math.Round(y.Level);
                                }
                            }
                        }
                    }
                    else patternAdded = false;
                } while (patternAdded);

                // solve final MIP
                opt.Defines["solveMasterAs"] = "MIP";
                masterJob.Run(opt, cutstockData);
                richTextBox1.AppendText("Optimal Solution: " + masterJob.OutDB.GetVariable("z").FindRecord().Level + Environment.NewLine); richTextBox1.ScrollToCaret();
                tabControl1.Visible = true;
                foreach (GAMSVariableRecord xp in masterJob.OutDB.GetVariable("xp"))
                {
                    if (xp.Level > 0.5)
                    {
                        richTextBox1.AppendText(String.Format("  pattern {0,2} {1,4} times: ", xp.Keys[0], Math.Round(xp.Level))); richTextBox1.ScrollToCaret();
                        GAMSParameterRecord aip = masterJob.OutDB.GetParameter("aip").FirstRecord(" ", xp.Keys[0]);
                        TabPage tp = new TabPage("pattern " + xp.Keys[0].ToString());
                        tp.BackColor = Color.White;
                        tabControl1.TabPages.Add(tp);

                        int x = 0;
                        double scale = tp.Width/masterJob.OutDB.GetParameter("r").FirstRecord().Value;
                        do
                        {
                            richTextBox1.AppendText(" " + aip.Keys[0] + ": " + aip.Value.ToString()); richTextBox1.ScrollToCaret();
                            // draw cuts
                            for (int i = 0; i < aip.Value; i++)
                            {
                                Panel p = new Panel();
                                p.BorderStyle = BorderStyle.FixedSingle;
                                p.BackColor = flowLayoutPanel1.Controls.Find(aip.Keys[0], true)[0].BackColor;
                                p.Height = tp.Height;
                                p.Width = (int) (width.FindRecord(aip.Keys[0]).Value * scale);
                                p.Left = x;
                                x += p.Width - 1;
                                tp.Controls.Add(p);
                            }
                        } while (aip.MoveNext());

                        Panel pExcess = new Panel();
                        pExcess.BorderStyle = BorderStyle.FixedSingle;
                        pExcess.BackColor = Color.LightGray;
                        pExcess.Height = tp.Height;
                        pExcess.Width = (int) (rawWidth.FirstRecord().Value*scale - x);
                        pExcess.Left = x;
                        tp.Controls.Add(pExcess);

                        richTextBox1.AppendText(Environment.NewLine); richTextBox1.ScrollToCaret();
                    }
                }
                // clean up of unmanaged resources
                cutstockData.Dispose();
                subMI.Dispose();
                opt.Dispose();
            }
            catch (Exception ex)
            {
                richTextBox1.AppendText("Exception: " + ex.Message);
            }
        }

        static String GetMasterModel()
        {
            String model = @"
$Title Cutting Stock - Master problem

Set  i    widths
Parameter
     w(i) width
     d(i) demand
Scalar
     r    raw width;
$gdxin csdata
$load i w d r

$if not set pmax $set pmax 1000
Set  p        possible patterns  /1*%pmax%/
     pp(p)    dynamic subset of p
Parameter
     aip(i,p) number of width i in pattern growing in p;

* Master model
Variable xp(p)     patterns used
         z         objective variable
Integer variable xp; xp.up(p) = sum(i, d(i));

Equation numpat    number of patterns used
         demand(i) meet demand;

numpat..     z =e= sum(pp, xp(pp));
demand(i)..  sum(pp, aip(i,pp)*xp(pp)) =g= d(i);

model master /numpat, demand/;";

            return model;
        }
        static String GetSubModel()
        {
            String submodel = @"
$Title Cutting Stock - Pricing problem is a knapsack model

Set  i    widths
Parameter
     w(i) width;
Scalar
     r    raw width;

$gdxin csdata
$load i w r

Parameter
     demdual(i) duals of master demand constraint /#i eps/;

Variable  z, y(i) new pattern;
Integer variable y; y.up(i) = ceil(r/w(i));

Equation defobj
         knapsack knapsack constraint;

defobj..     z =e= 1 - sum(i, demdual(i)*y(i));
knapsack..   sum(i, w(i)*y(i)) =l= r;
option optcr=0;
model pricing /defobj, knapsack/; pricing.optfile=1";

            return submodel;
        }

        private void button4_Click(object sender, EventArgs e)
        {
            OpenFileDialog browser = new OpenFileDialog();

            browser.DefaultExt = "accdb";
            browser.Filter = "Access Database (*.accdb)|*.accdb";
            browser.InitialDirectory = Environment.CurrentDirectory;

            if (browser.ShowDialog() == DialogResult.OK)
            {
                // read from access database
                OleDbConnection connection = null;
                try
                {
                    connection = new OleDbConnection("Provider=Microsoft.ACE.OLEDB.12.0;Data Source=" + browser.FileName);

                    // delete cut widths
                    while (nrCuts>0)
                    {
                        flowLayoutPanel1.Controls.RemoveAt(nrCuts-1);
                        nrCuts--;
                    }
                    flowLayoutPanel1.Update();

                    OleDbCommand cmd = new OleDbCommand("SELECT Width FROM RawWidth", connection);
                    connection.Open();
                    OleDbDataReader reader = cmd.ExecuteReader();
                    while (reader.Read())
                    {
                        nudWidth.Value = (int)reader.GetValue(0);
                        nudWidth.Update();
                    }

                    cmd = new OleDbCommand("SELECT Label, Width, Demand FROM CutWidths", connection);
                    reader = cmd.ExecuteReader();
                    while (reader.Read()) {
                        flowLayoutPanel1.Controls.Add(new CutWidthsControl(reader.GetString(0), (int)reader.GetValue(1), (int)nudWidth.Value, (int)reader.GetValue(2), colors[nrCuts % colors.Count], nudWidth));
                        flowLayoutPanel1.VerticalScroll.Value = flowLayoutPanel1.VerticalScroll.Maximum;
                        flowLayoutPanel1.Update();
                        flowLayoutPanel1.VerticalScroll.Value = flowLayoutPanel1.VerticalScroll.Maximum;
                        nrCuts++;
                    }
                    connection.Close();
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Error reading data from database. \n" + ex.Message);
                    connection.Close();
                    Environment.Exit(1);
                }
            }

        }
    }
}
