﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using GAMS;

namespace BendersDecomposition2StageSP
{
    class Benders2Stage
    {
        static void Main(string[] args)
        {
            GAMSWorkspace ws = new GAMSWorkspace();
            GAMSJob data = ws.AddJobFromString(GetDataText());
            data.Run();
            GAMSParameter ScenarioData = data.OutDB.GetParameter("ScenarioData");

            GAMSOptions opt = ws.AddOptions();
            opt.Defines.Add("datain", data.OutDB.Name);
            int maxiter = 25;
            opt.Defines.Add("maxiter", maxiter.ToString());
            opt.AllModelTypes = "cplexd";
            opt.OptFile = 1;

            GAMSCheckpoint cpMaster = ws.AddCheckpoint();
            GAMSCheckpoint cpSub = ws.AddCheckpoint();

            using (GAMSJob master = ws.AddJobFromString(GetMasterText()))
            {
                master.Run(opt, cpMaster, data.OutDB);
            }
            GAMSModelInstance masteri = cpMaster.AddModelInstance();
            GAMSParameter cutconst = masteri.SyncDB.AddParameter("cutconst", 1, "Benders optimality cut constant");
            GAMSParameter cutcoeff = masteri.SyncDB.AddParameter("cutcoeff", 2, "Benders optimality coefficients");
            GAMSVariable theta = masteri.SyncDB.AddVariable("theta", 0, VarType.Free, "Future profit function variable");
            masteri.Instantiate("masterproblem max zmaster using lp", opt, cutconst, cutcoeff, theta);

            using (GAMSJob sub = ws.AddJobFromString(GetSubText()))
            {
                sub.Run(opt, cpSub, data.OutDB);
            }
            GAMSModelInstance subi = cpSub.AddModelInstance();
            GAMSParameter received = subi.SyncDB.AddParameter("received", 1, "units received from master");
            GAMSParameter demand = subi.SyncDB.AddParameter("demand", 1, "stochastic demand");
            subi.Instantiate("subproblem max zsub using lp", opt, received, demand);

            opt.Dispose();

            double lowerbound = double.NegativeInfinity, upperbound = double.PositiveInfinity, objmaster = double.PositiveInfinity;
            int iter = 1;
            do
            {
                Console.WriteLine("Iteration: " + iter);
                // Solve master
                if (1 == iter) // fix theta for first iteration
                {
                    GAMSVariableRecord thetaRec = theta.AddRecord();
                    thetaRec.Lower = 0;
                    thetaRec.Upper = 0;
                }
                else
                    theta.Clear();

                masteri.Solve(GAMSModelInstance.SymbolUpdateType.BaseCase);
                Console.WriteLine(" Master " + masteri.ModelStatus + " : obj=" + masteri.SyncDB.GetVariable("zmaster").FirstRecord().Level);
                if (1 < iter)
                    upperbound = masteri.SyncDB.GetVariable("zmaster").FirstRecord().Level;
                objmaster = masteri.SyncDB.GetVariable("zmaster").FirstRecord().Level - theta.FirstRecord().Level;

                // Set received from master
                received.Clear();
                foreach (GAMSVariableRecord r in masteri.SyncDB.GetVariable("received"))
                {
                    received.AddRecord(r.Keys).Value = r.Level;
                    cutcoeff.AddRecord(iter.ToString(), r.Keys[0]);
                }

                cutconst.AddRecord(iter.ToString());
                double objsub = 0.0;
                foreach (GAMSSetRecord s in data.OutDB.GetSet("s"))
                {
                    demand.Clear();
                    foreach (GAMSSetRecord j in data.OutDB.GetSet("j"))
                        demand.AddRecord(j.Keys).Value = ScenarioData.FindRecord(s.Keys[0], j.Keys[0]).Value;

                    subi.Solve(GAMSModelInstance.SymbolUpdateType.BaseCase);
                    Console.WriteLine(" Sub " + subi.ModelStatus + " : obj=" + subi.SyncDB.GetVariable("zsub").FirstRecord().Level);


                    double probability = ScenarioData.FindRecord(s.Keys[0], "prob").Value;
                    objsub += probability * subi.SyncDB.GetVariable("zsub").FirstRecord().Level;
                    foreach (GAMSSetRecord j in data.OutDB.GetSet("j"))
                    {
                        cutconst.FindRecord(iter.ToString()).Value += probability * subi.SyncDB.GetEquation("market").FindRecord(j.Keys).Marginal * demand.FindRecord(j.Keys).Value;
                        cutcoeff.FindRecord(iter.ToString(), j.Keys[0]).Value += probability * subi.SyncDB.GetEquation("selling").FindRecord(j.Keys).Marginal;
                    }
                }
                lowerbound = Math.Max(lowerbound, objmaster + objsub);
                iter++;
                if (iter == maxiter + 1)
                    throw new Exception("Benders out of iterations");

                Console.WriteLine(" lowerbound: " + lowerbound + " upperbound: " + upperbound + " objmaster: " + objmaster);
            } while ((upperbound - lowerbound) >= 0.001 * (1 + Math.Abs(upperbound)));

            masteri.Dispose();
            subi.Dispose();
            data.Dispose();
        }

        static String GetDataText()
        {
            String model = @"
Sets
   i factories                                   /f1*f3/
   j distribution centers                        /d1*d5/

Parameter
   capacity(i) unit capacity at factories
                 /f1 500, f2 450, f3 650/
   demand(j)   unit demand at distribution centers
                 /d1 160, d2 120, d3 270, d4 325, d5 700 /
   prodcost    unit production cost                   /14/
   price       sales price                            /24/
   wastecost   cost of removal of overstocked products /4/

Table transcost(i,j) unit transportation cost
       d1    d2    d3    d4    d5
  f1   2.49  5.21  3.76  4.85  2.07
  f2   1.46  2.54  1.83  1.86  4.76
  f3   3.26  3.08  2.60  3.76  4.45;

Set
  s scenarios /lo,mid,hi/

table ScenarioData(s,*) possible outcomes for demand plus probabilities
     d1  d2  d3  d4  d5 prob
lo  150 100 250 300 600 0.25
mid 160 120 270 325 700 0.50
hi  170 135 300 350 800 0.25;
";

            return model;
        }

        static String GetMasterText()
        {
            String model = @"
Sets
   i factories
   j distribution centers

Parameter
   capacity(i)    unit capacity at factories
   prodcost       unit production cost
   transcost(i,j) unit transportation cost

$if not set datain $abort 'datain not set'
$gdxin %datain%
$load i j capacity prodcost transcost

* Benders master problem
$if not defined maxiter $set maxiter 25
Set
   iter             max Benders iterations /1*%maxiter%/

Parameter
   cutconst(iter)   constants in optimality cuts
   cutcoeff(iter,j) coefficients in optimality cuts

Variables
   ship(i,j)        shipments
   product(i)       production
   received(j)      quantity sent to market
   zmaster          objective variable of master problem
   theta            future profit
Positive Variables ship;

Equations
   masterobj        master objective function
   production(i)    calculate production in each factory
   receive(j)       calculate quantity to be send to markets
   optcut(iter)     Benders optimality cuts;

masterobj..
    zmaster =e=  theta -sum((i,j), transcost(i,j)*ship(i,j))
                       - sum(i,prodcost*product(i));

receive(j)..       received(j) =e= sum(i, ship(i,j));

production(i)..    product(i) =e= sum(j, ship(i,j));
product.up(i) = capacity(i);

optcut(iter)..  theta =l= cutconst(iter) +
                           sum(j, cutcoeff(iter,j)*received(j));

model masterproblem /all/;

* Initialize cut to be non-binding
cutconst(iter) = 1e15;
cutcoeff(iter,j) = eps;
";

            return model;
        }

        static String GetSubText()
        {
            String model = @"
Sets
   i factories
   j distribution centers

Parameter
   demand(j)   unit demand at distribution centers
   price       sales price
   wastecost   cost of removal of overstocked products
   received(j) first stage decision units received

$if not set datain $abort 'datain not set'
$gdxin %datain%
$load i j demand price wastecost

* Benders' subproblem

Variables
   sales(j)         sales (actually sold)
   waste(j)         overstocked products
   zsub             objective variable of sub problem
Positive variables sales, waste

Equations
   subobj           subproblem objective function
   selling(j)       part of received is sold
   market(j)        upperbound on sales
;

subobj..
   zsub =e= sum(j, price*sales(j)) - sum(j, wastecost*waste(j));

selling(j)..  sales(j) + waste(j) =e= received(j);

market(j)..   sales(j) =l= demand(j);

model subproblem /subobj,selling,market/;

* Initialize received
received(j) = demand(j);
";

            return model;
        }

    }
}
