﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GAMS;

namespace Step1
{
    class Step1
    {
        static void Main(string[] args)
        {
            GAMSWorkspace ws = new GAMSWorkspace();

            GAMSDatabase cutstockData = ws.AddDatabase();
            GetModelData(cutstockData);

            GAMSJob cut = ws.AddJobFromString(GetModelText());

            GAMSOptions opt = ws.AddOptions();
            opt.AllModelTypes = "Gurobi";
            opt.Defines.Add("gdxincname", cutstockData.Name);

            try
            {
                cut.Run(opt, Console.Out, cutstockData);

                foreach (GAMSParameterRecord rep in cut.OutDB.GetParameter("patrep"))
                {
                    Console.WriteLine(rep.Keys[0] + ", pattern " + rep.Keys[1] + ": " + rep.Value);
                }
            }
            catch (GAMSException e)
            {
                Console.WriteLine("Problem in GAMS: " + e.Message);
                return;
            }
            catch (System.Exception e)
            {
                Console.WriteLine("System Error: " + e.Message);
                return;
            }

        }

        static void GetModelData(GAMSDatabase cutstockData)
        {
            Dictionary<string, double> d = 
                new Dictionary<string, double>() { { "i1", 97 }, { "i2", 610 }, { "i3", 395 }, { "i4", 211 } };
            Dictionary<string, double> w = 
                new Dictionary<string, double>() { { "i1", 47 }, { "i2", 36 }, { "i3", 31 }, { "i4", 14 } };
            int r = 100; // raw width

            GAMSSet widths = cutstockData.AddSet("i", 1, "widths");
            GAMSParameter rawWidth = cutstockData.AddParameter("r", 0, "raw width");
            GAMSParameter demand = cutstockData.AddParameter("d", 1, "demand");
            GAMSParameter width = cutstockData.AddParameter("w", 1, "width");

            rawWidth.AddRecord().Value = r;
            foreach (string i in d.Keys)
                widths.AddRecord(i);
            foreach (KeyValuePair<string, double> t in d)
                demand.AddRecord(t.Key).Value = t.Value;
            foreach (KeyValuePair<string, double> t in w)
                width.AddRecord(t.Key).Value = t.Value;
        }

        static String GetModelText()
        {
            String model = @"
$Title Cutting Stock - A Column Generation Approach (CUTSTOCK,SEQ=294)

$ontext
 The task is to cut out some paper products of different sizes from a
 large raw paper roll, in order to meet a customer's order. The objective
 is to minimize the required number of paper rolls.


P. C. Gilmore and R. E. Gomory, A linear programming approach to the
cutting stock problem, Part I, Operations Research 9 (1961), 849-859.

P. C. Gilmore and R. E. Gomory, A linear programming approach to the
cutting stock problem, Part II, Operations Research 11 (1963), 863-888.
$offtext

Set  i    widths
Parameter
     r    raw width
     w(i) width
     d(i) demand ;

$if not set gdxincname $abort 'no include file name for data file provided'
$gdxin %gdxincname%
$load r i w d
$gdxin

* Gilmore-Gomory column generation algorithm

Set  p        possible patterns  /p1*p1000/
     pp(p)    dynamic subset of p
Parameter
     aip(i,p) number of width i in pattern growing in p;


* Master model
Variable xp(p)     patterns used
         z         objective variable
Integer variable xp; xp.up(p) = sum(i, d(i));

Equation numpat    number of patterns used
         demand(i) meet demand;

numpat..     z =e= sum(pp, xp(pp));
demand(i)..  sum(pp, aip(i,pp)*xp(pp)) =g= d(i);

model master /numpat, demand/;

* Pricing problem - Knapsack model
Variable  y(i) new pattern;
Integer variable y; y.up(i) = ceil(r/w(i));

Equation defobj
         knapsack knapsack constraint;

defobj..     z =e= 1 - sum(i, demand.m(i)*y(i));
knapsack..   sum(i, w(i)*y(i)) =l= r;

model pricing /defobj, knapsack/;

* Initialization - the initial patterns have a single width
pp(p) = ord(p)<=card(i);
aip(i,pp(p))$(ord(i)=ord(p)) = floor(r/w(i));
*display aip;

Scalar done  loop indicator /0/
Set    pi(p) set of the last pattern; pi(p) = ord(p)=card(pp)+1;

option optcr=0,limrow=0,limcol=0,solprint=off;

While(not done and card(pp)<card(p),
   solve master using rmip minimizing z;
   solve pricing using mip minimizing z;

* pattern that might improve the master model found?
   if(z.l < -0.001,
      aip(i,pi) = round(y.l(i));
      pp(pi) = yes; pi(p) = pi(p-1);
   else
      done = 1;
   );
);
display 'lower bound for number of rolls', master.objval;

option solprint=on;
solve master using mip minimizing z;

Parameter patrep Solution pattern report
          demrep Solution demand supply report;

patrep('# produced',p) = round(xp.l(p));
patrep(i,p)$patrep('# produced',p) = aip(i,p);
patrep(i,'total') = sum(p, patrep(i,p));
patrep('# produced','total') = sum(p, patrep('# produced',p));

demrep(i,'produced') = sum(p,patrep(i,p)*patrep('# produced',p));
demrep(i,'demand') = d(i);
demrep(i,'over') = demrep(i,'produced') - demrep(i,'demand');

display patrep, demrep;
";

            return model;
        }

    }
}
