﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GAMS;
using System.Windows.Forms.DataVisualization.Charting;
using System.Diagnostics;
using System.Data.OleDb;
using System.Reflection;

namespace TransportGUI
{
    public partial class Form1 : Form
    {
        private Dictionary<Tuple<string, string>, double> distance = new Dictionary<Tuple<string, string>, double>() 
            { 
                { new Tuple<string,string> ("Seattle",   "New-York"), 2.5 },
                { new Tuple<string,string> ("Seattle",   "Chicago"),  1.7 },
                { new Tuple<string,string> ("Seattle",   "Topeka"),   1.8 },
                { new Tuple<string,string> ("San-Diego", "New-York"), 2.5 },
                { new Tuple<string,string> ("San-Diego", "Chicago"),  1.8 },
                { new Tuple<string,string> ("San-Diego", "Topeka"),   1.4 }
            };

        private Dictionary<string, double> capacity = new Dictionary<string, double>()
            {
                { "Seattle",   350},
                { "San-Diego", 600}
            };

        private Dictionary<string, double> demand = new Dictionary<string, double>()
            {
                { "New-York",   325},
                { "Chicago",    300},
                { "Topeka",     275}
            };

        private double bparMin   = 0.2;
        private double bparMax   = 1.0;
        private double bparSteps = 0.05;

        public Form1()
        {
            InitializeComponent();
            this.chart1.Series.Clear();
            this.chart1.Visible = false;
            // bind default data to data grids
            foreach (Tuple<string, string> t in distance.Keys)
                this.dataGridDistance.Rows.Add(new string[]{ t.Item1, t.Item2, distance[t].ToString()} );
            foreach (string t in capacity.Keys)
                this.dataGridCapacity.Rows.Add(new string[] { t, capacity[t].ToString() });
            foreach (string t in demand.Keys)
                this.dataGridDemand.Rows.Add(new string[] { t, demand[t].ToString() });
            this.dataGridBpar.Rows.Add(new string[] {bparMin.ToString(), bparMax.ToString(), bparSteps.ToString() });
        }

        private void button1_Click(object sender, EventArgs e)
        {
            this.dataGridView1.Rows.Clear();
            this.chart1.Series.Clear();
            this.chart1.ResetAutoValues();
            this.chart1.Visible = true;
            this.chart1.Series.Add("cost");

            GAMSWorkspace ws = new GAMSWorkspace(@"C:\tmp\transport");

            // get data from user input
            GAMSDatabase data = ws.AddDatabase("data");

            GAMSParameter d = data.AddParameter("d", 2, "distance in thousands of miles");
            foreach (DataGridViewRow row in this.dataGridDistance.Rows)
                d.AddRecord(row.Cells[0].Value.ToString(), row.Cells[1].Value.ToString()).Value = Convert.ToDouble(row.Cells[2].Value);

            GAMSSet i = data.AddSet("i", 1, "canning plants");
            GAMSParameter a = data.AddParameter("a", 1, "capacity of plant i in cases");
            foreach (DataGridViewRow row in this.dataGridCapacity.Rows)
            {
                i.AddRecord(row.Cells[0].Value.ToString());
                a.AddRecord(row.Cells[0].Value.ToString()).Value = Convert.ToDouble(row.Cells[1].Value);
            }

            GAMSSet j = data.AddSet("j", 1, "markets");
            GAMSParameter b = data.AddParameter("b", 1, "demand at market j in cases");
            foreach (DataGridViewRow row in this.dataGridDemand.Rows)
            {
                j.AddRecord(row.Cells[0].Value.ToString());
                b.AddRecord(row.Cells[0].Value.ToString()).Value = Convert.ToDouble(row.Cells[1].Value);
            }

            // initialize a GAMSCheckpoint by running a GAMSJob
            GAMSOptions opt = ws.AddOptions();
            opt.AllModelTypes = "conopt";
            GAMSJob t7 = ws.AddJobFromString(GetModelText());
            GAMSCheckpoint cp = ws.AddCheckpoint();
            t7.Run(opt, cp, data);

            // create a GAMSModelInstance and solve it multiple times with different scalar bpar
            GAMSModelInstance mi = cp.AddModelInstance();

            GAMSParameter bpar = mi.SyncDB.AddParameter("bpar", 0, "demand multiplier");

            // instantiate the GAMSModelInstance and pass a model definition and GAMSModifier to declare bpar mutable
            mi.Instantiate("transport us nlp min z", opt, new GAMSModifier(bpar));

            bpar.AddRecord();
            List<double> bparlist = new List<double>();
            double v = Convert.ToDouble(this.dataGridBpar.Rows[0].Cells[0].Value);  // bmultMin
            do
            {
                bparlist.Add(v);
                v += Convert.ToDouble(this.dataGridBpar.Rows[0].Cells[2].Value);    // v+= bmultSteps
            } while (v < Convert.ToDouble(this.dataGridBpar.Rows[0].Cells[1].Value) + 1e-10);  // v <= bmultMax 

            foreach (double bm in bparlist)
            {
                bpar.FirstRecord().Value = bm;
                mi.Solve();
                double obj = mi.SyncDB.GetVariable("z").FindRecord().Level;
                string[] row = new string[] { bm.ToString("0.00"), mi.ModelStatus.ToString(), mi.SolveStatus.ToString(), obj.ToString("0.00") };
                this.dataGridView1.Rows.Add(row);

                if (mi.ModelStatus == ModelStat.OptimalGlobal || mi.ModelStatus == ModelStat.OptimalLocal)
                {
                    this.chart1.Series["cost"].Points.AddXY(bm, obj);
                    this.chart1.Update();
                }
            }
        }

        private void button2_Click(object sender, EventArgs e)
        {
            OpenFileDialog browser = new OpenFileDialog();

            browser.DefaultExt = "accdb";
            browser.Filter = "Access Database (*.accdb)|*.accdb";
            browser.InitialDirectory = Environment.CurrentDirectory;

            if (browser.ShowDialog() == DialogResult.OK)
            {
                // read from access database
                OleDbConnection connection = null;
                try
                {
                    connection = new OleDbConnection("Provider=Microsoft.ACE.OLEDB.12.0;Data Source=" + browser.FileName);

                    // read distance
                    this.dataGridDistance.Rows.Clear();
                    OleDbCommand cmd = new OleDbCommand("SELECT Plant, Market, Distance FROM Distance", connection);
                    connection.Open();
                    OleDbDataReader reader = cmd.ExecuteReader();
                    while (reader.Read())
                        this.dataGridDistance.Rows.Add(new string[] { reader.GetString(0), reader.GetString(1), reader.GetValue(2).ToString() });

                    // read capacity
                    this.dataGridCapacity.Rows.Clear();
                    cmd = new OleDbCommand("SELECT Plant, Capacity FROM Plant", connection);
                    reader = cmd.ExecuteReader();
                    while (reader.Read())
                        this.dataGridCapacity.Rows.Add(new string[] { reader.GetString(0), reader.GetValue(1).ToString() });

                    // read demand
                    this.dataGridDemand.Rows.Clear();
                    cmd = new OleDbCommand("SELECT Market, Demand FROM Market", connection);
                    reader = cmd.ExecuteReader();
                    while (reader.Read())
                        this.dataGridDemand.Rows.Add(new string[] { reader.GetString(0), reader.GetValue(1).ToString() });
                    connection.Close();
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Error reading data from database. \n" + ex.Message);
                    connection.Close();
                    Environment.Exit(1);
                }
            }
        }

        static String GetModelText()
        {
            String model = @"
  Sets
       i   canning plants
       j   markets;

  Parameters

       a(i)  capacity of plant i in cases

       b(j)  demand at market j in cases

       d(i,j)  distance in thousands of miles;

$gdxin data
$load i j a b d


  Scalar f      freight in dollars per case per thousand miles  /90/ ;
  Scalar bpar   demand parameter /1/;

  Parameter c(i,j)  transport cost in thousands of dollars per case ;

            c(i,j) = f * d(i,j) / 1000 ;

  Variables
       x(i,j)  shipment quantities in cases
       z       total transportation costs in thousands of dollars ;

  Positive Variable x ;

  Equations
       cost        define objective function
       supply(i)   observe supply limit at plant i
       demand(j)   satisfy demand at market j ;

  cost ..        z  =e=  sum((i,j), c(i,j)*x(i,j)**bpar) ;

  supply(i) ..   sum(j, x(i,j))  =l=  a(i) ;

  demand(j) ..   sum(i, x(i,j))  =g=  sqr(1-bpar)*b(j) ;

  Model transport /all/ ;
";

            return model;
        }
    }
}
