$Title Combined Heat and Power Generation
$eolcom //

$if not set maxh $set maxh 672
Sets  h           Hours                    / h1*h%maxh% /
      m           Matter                   / Gas, Coal, Wasteheat, Exhaust, Steam, Heat, Electricity /
      mext(m)     External demanded Matter / Heat, Electricity /
      fuel(m)     Fuel                     / Gas, Coal /
      p           Processes                / SG  SteamGenerator
                                             HTB HighTemperatureBoiler
                                             GT  GasTurbine
                                             HRB HeatRecoveryBoiler
                                             HE  HeatExchanger
                                             HB  HeatBypass
                                             ST  SteamTurbine
                                             HC  HeatCondenser /
      pmin(p)     Processes with minimum utilization level
      s           Processsegment           / 0, 1 /
      tsh         Time series header       / HeatDemand, ElectricityDemand, EXPrice /;

Parameter
      ts(h,tsh)    Time series
      dem(h,mext)  External demand in MW (heat and electricity)
      ex(h)        Energy eXchange electricity price in $ per MW
      io(m,p,s)    In- and outputprocesstable
      fp(fuel)     Fuel price in $ per MWh                                         / Gas  19.23
                                                                                     Coal 12.23 /
      startcost(p) Startup costs in $                                              / SG 1000, GT 500  /;

Scalar
      chpbonus     Bonus for electricity generation with cogeneration in $ per MWh / 8 /
      hbcosts      Costs for using heat bypass in $ per MWh                        / 4 /
      hloss        Heat loss in storage tank in % per h                            / 2 /
      pumpcost     Costs for filling up heat storage tank in $ per MWh             / 0.05 /
      maxgtstart   Maximum number of allowed GT starts                             / 8 /;

$call gdxxrw chpin2.xls o=chpin.gdx par=io rng=Plant!a1 rdim=1 cdim=2 par=ts rng=Timeseries!a1 dim=2
$if errorlevel 1 $abort 'Problem when reading Exceldata'

$gdxin chpin
$load ts io

dem(h,'Electricity') = ts(h,'ElectricityDemand');
dem(h,'Heat')        = ts(h,'HeatDemand');
ex(h)                = ts(h,'EXPrice');
pmin(p)              = sum(m, io(m,p,'0')) <> 0;

Variable
      PL(h,p,s)        Processlevel
      F(h,m)           Fueluse in MW
      G(h,m)           Generated heat and electricity in MW
      TEX(h)           Trade of electricity at EX in MW
      ONOFF(h,p)       Indicator for running processes
      STARTUP(h,p)     Indicator for start of processes
      SHUTDOWN(h,p)    Indicator for end of processes
      HIN(h)           Input in HST in MW
      HOUT(h)          Output from HST in MW
      HLVL(h)          Level of HST at the end of an hour in MW
      OBJ              Systemcost in $;

Positive Variables PL, F, G, HIN, HOUT, HLVL;
Binary Variables ONOFF, STARTUP, SHUTDOWN;

Equations
      defobj           Objective function
      defbal(h,m)      Mass balance
      defE(h)          Electricity balance
      defH(h)          Heat balance
      defHSTlvl        HST balance
      defseg(h,p)      Convexcombination of segments
      defstartup(h,p)  Process start
      defGTstartmax    Maximum number of GT startups
      defshutdownGT(h) GT turned off
      defGToff(h)      GT has to be off for 8 hours after shutdown;

defobj..     OBJ =e=  sum((h,fuel), fp(fuel)*F(h,fuel))           // Fuelcosts
                    + sum(h,        ex(h)*TEX(h))                 // Costs/returns from electricity trading
                    - sum(h,        chpbonus*G(h,'Electricity'))  // Bonus for electricity generation in cogeneration
                    + sum((h,s),    hbcosts*(-1)*io('Wasteheat','HB',s)*PL(h,'HB',s)) // Extracosts
                    + sum(h,        pumpcost*HIN(h))              // Pumpcosts for filling HST
                    + sum((h,p)$startcost(p), startcost(p)*STARTUP(h,p));             // Startupcosts


defbal(h,m)..  0 =e= F(h,m)$fuel(m) - G(h,m)$mext(m) +
                     sum((p,s)$io(m,p,s), io(m,p,s)*PL(h,p,s));

defE(h)..      dem(h,'Electricity')  =e= G(h,'Electricity') + TEX(h);

defH(h)..      dem(h,'Heat') =e= G(h,'Heat') + HOUT(h) - HIN(h);

defHSTlvl(h).. HLVL(h) =e= HLVL(h-1)*(1-hloss/100) - HOUT(h) + HIN(h);

defseg(h,p)..  sum(s, PL(h,p,s)) =e= 1$(not pmin(p)) + ONOFF(h,p)$pmin(p);

defstartup(h,p)$startcost(p)..  STARTUP(h,p) =g= ONOFF(h,p) - ONOFF(h-1,p);

defGTstartmax..  sum(h,STARTUP(h,'GT')) =l= maxgtstart;

defshutdownGT(h)..  SHUTDOWN(h,'GT') =g= ONOFF(h-1,'GT') - ONOFF(h,'GT');

alias (h,hh);
defGTOff(h).. sum(hh$(ord(hh)>=ord(h) and ord(hh)<ord(h)+8), STARTUP(hh,'GT')) =l= 1-SHUTDOWN(h,'GT');

HLVL.up(h) = 50;
HIN.up(h)  = 15;
HOUT.up(h) = 12;

model chp /all/;

solve  chp min obj using mip;

parameter rep;
rep('GT ON')     = sum(h, ONOFF.l(h,'GT'));
rep('GT Starts') = sum(h, max(0,ONOFF.l(h,'GT')-ONOFF.l(h-1,'GT')));
display rep;
