$TITLE Optimal Design for Chemical Batch Processing using Outer Approximation

$Ontext

   A chemical batch process having three processing stages: mixing,
   reaction and centrifuge separation, produce two products. This
   model is used to determine the number and sizes of units to
   operate in parallel at each stage. The resulting model is
   nonlinear and mixed integer;

   This model implements a simple Outer Approximation Scheme. The
   linearization is done using the Convert option 'Jacobian' and a
   Python script to transform the scalar structure of the Convert
   jacobian.gdx into an index version.

Morari, M, and Grossmann, I E, Eds, Chemical Engineering Optimization
Models with GAMS. Computer Aids for Chemical Engineering Corporation,
1991.

Kocis, G R, and Grossmann, I E, Global Optimization of Nonconvex MINLP
Problems in Process Synthesis. Independent Engineering Chemical
Research 27 (1988), 1407-1421.

$Offtext
Sets         i         products         / a , b /
             j         stages           /mixer, reactor, centrifuge/
             k         potential number of parallel units  / 1 * 3 /

Scalar       h         horizon time (available time hrs)  /6000./
             vlow      lower bound for size of batch unit /250./
             vupp      upper bound for size of batch unit /2500./

Parameters   q(i)      demand of product i  /
                        a   200000
                        b   150000        /
             alpha(j)  cost coefficient for batch units
                      / mixer        250.
                        reactor      500.
                        centrifuge   340. /
             beta(j)   cost exponent for batch units
                      / mixer        0.6
                        reactor      0.6
                        centrifuge   0.6  /

* the parameters below are defined to obtain the original variables
             actv(j)   actual volume (l)
             actb(i)   actual batch sizes (kg)
             acttl(i)  actual cycle times (hrs)
             actn(j)   actual number of units in parallel
             coeff(k)  represent number of parallel units   ;

coeff(k) = log(ord(k))  ;

Table        s(i,j)    size factor for product i in stage j (kg per l)
           mixer     reactor   centrifuge
   a          2         3         4
   b          4         6         3

Table        t(i,j)    processing time of product i in batch j (hrs)
           mixer     reactor   centrifuge
   a          8        20         4
   b         10        12         3

Variables    y(k,j)    binary variable denoting stage existence
             v(j)      volume of stage j (l)
             b(i)      batch size of product i (kg)
             tl(i)     cycle time of product i (hrs)
             n(j)      number of units in parallel stage j
             cost      total cost of batch processing units ($)  ;

Binary   Variables     y(k,j) ;
Positive Variables     v(j) , b(i) , tl(i) , n(j) ;

Equations    vol(i,j)   calculate volume of stage j
             cycle(i,j) calculate cycle time of product i
             time       time constraint
             units(j)   calculate number of processing units per stage
             lim(j)     limit selection to one number
             obj        objective function definition ;

* convexified formulation of model equations
* volume requirement in stage j
vol(i,j)..   v(j)  =g=  log(s(i,j)) +b(i) ;

* cycle time for each product i
cycle(i,j).. n(j) + tl(i)  =g=  log(t(i,j)) ;

* constraint for production time
time..       sum(i , q(i) * exp(tl(i) - b(i))) =l= h ;

* relating number of units to 0-1 variables
units(j)..   n(j)  =e=       sum(k , coeff(k) * y(k,j)) ;

* only one choice for parallel units is feasible
lim(j)..     sum(k , y(k,j)) =e= 1 ;

* defining objective function
obj..        cost  - sum(j ,alpha(j)*(exp(n(j) + beta(j)*v(j)))) =g= 0;

Model   batch    / all /

Set          l / 1*5 /
             tlin(l)       linearizations time
             olin(l)       linearizations objective
             sc(l)         solution cut
             sce(l,k,j)    solution elements

Parameter    tcoef_tl(l,i) coefficient of tl in linearized time
             tcoef_b(l,i)  coefficient of b in linearized time
             trhs(l)       right hand side of linearized time
             ocoef_n(l,j)  coefficient of n in linearized objective
             ocoef_v(l,j)  coefficient of b in linearized objective
             orhs(l)       right hand side of linearized objective

Equations    time_l(l) linearizations time
             obj_l(l)  linearizations objective
             solcut(l) solution cutting;

time_l(tlin(l)).. sum(i, tcoef_tl(l,i)*tl(i) + tcoef_b(l,i)*b(i)) =l= trhs(l);

obj_l(olin(l)).. cost + sum(j, ocoef_n(l,j)*n(j) + ocoef_v(l,j)*v(j)) =g= orhs(l);

solcut(sc(l)).. sum((k,j)$sce(l,k,j),           y(k,j)) +
                sum((k,j)$(not sce(l,k,j)), 1 - y(k,j)) =l= card(k)*card(j)-1;

Model   batchmip / vol, time_l, cycle, units, lim, obj_l, solcut /;
tlin(l) = no;
olin(l) = no;

* bounds section
v.lo(j)   =  log(vlow) ;
v.up(j)   =  log(vupp);
n.up(j)   =  log(3.) ;

* tight lower bounds are computed below for cycle times and batch sizes
Parameters   tllow(i)       lower bound on tl(i)
             tlupp(i)       upper bound on tl(i) ;

tllow(i)  =  smax(j, t(i,j) / exp(n.up(j))) ;
tlupp(i)  =  smax(j, t(i,j) ) ;

tl.lo(i)  =  log(tllow(i)) ;
tl.up(i)  =  log(tlupp(i)) ;

Parameters   blow(i)        lower bound on b(i)
             bupp(i)        upper bound on b(i)  ;

blow(i)  =  q(i) * ( smax(j, t(i,j) / exp(n.up(j))))/ h;
bupp(i)  =  min( q(i) , smin(j , exp(v.up(j))/s(i,j))) ;

b.lo(i)   =  log(blow(i)) ;
b.up(i)   =  log(bupp(i)) ;

*  initial point
* binary variables set for 3 units per stage

y.l(k,j)              = 0.;
y.l('3','mixer')      = 1;
y.l('3','reactor')    = 1;
y.l('3','centrifuge') = 1;
n.l(j)                = sum( k , coeff(k)*  y.l(k,j) ) ;

* batch sizes are set at the mid-point between bounds
b.l(i)    = (b.up(i) + b.lo(i))/2 ;
v.l(j)    = smax(i ,  log(s(i,j)) + b.l(i)) ;
tl.l(i)   = smax(j ,  log(t(i,j))-n.l(j)) ;

*solve batch min cost using minlp;
*exit

$onecho > convert.opt
jacobian
dict
$offecho

Parameter A(*,*,*,*,*,*) /
  time.''.''.tl.#i.'' 0
  time.''.''.b .#i.'' 0
  obj .''.''.v .#j.'' 0
  obj .''.''.n .#j.'' 0
/;

Parameter rep;

Option optcr=0, limrow=0, limcol=0, solprint=silent, solvelink=5;
option rminlp=conopt; batch.optfile=0; Solve batch using rminlp minimizing cost ;
option rminlp=convert; batch.optfile=1; Solve batch using rminlp minimizing cost ;

rep('1','NLP') = cost.l;

scalar goon /1/, bestobj /inf/; sc(l)=no; sce(l,k,j)=no;
alias (l, ll);

$set pysrc .\
loop(ll$goon,
  tlin(ll) = yes;
  execute '%pysrc%jacobian.py %gams.sysdir%';
  abort$errorlevel 'something wrong with jacobian.py';
  execute_load 'jac_out.gdx', A;
*  display A;
  tcoef_tl(ll,i) = A('time','','','tl',i,'');
  tcoef_b (ll,i) = A('time','','','b' ,i,'');
  trhs(ll) = h - time.l + sum(i, tcoef_tl(ll,i)*tl.l(i) + tcoef_b(ll,i)*b.l(i));

  olin(ll) = yes;
  ocoef_v (ll,j) = A('obj','','','v' ,j,'');
  ocoef_n (ll,j) = A('obj','','','n' ,j,'');
  orhs(ll) = 0 - obj.l + cost.l + sum(j, ocoef_n(ll,j)*n.l(j) + ocoef_v(ll,j)*v.l(j));

  Solve batchmip using mip minimizing cost ;

  sc(ll) = yes;
  sce(ll,k,j) = round(y.l(k,j));

  if (batchmip.modelstat=1 or batchmip.modelstat=8,
    rep(ll,'MIP') = cost.l;
    y.fx(k,j) = round(y.l(k,j));
    option rminlp=conopt; batch.optfile=0; Solve batch using rminlp minimizing cost ;
    if ((batch.modelstat=1 or batch.modelstat=2) and cost.l < bestobj,
      bestobj = cost.l;
    )
    option rminlp=convert; batch.optfile=1; Solve batch using rminlp minimizing cost ;
    rep(ll+1,'NLP') = cost.l;
    y.lo(k,j) = 0; y.up(k,j) = 1;
  else
    rep(ll,'MIP') = na;
    goon = 0));

option rep:2:0:1; display bestobj, rep;
